{  $Id: ncmodem.pas 6984 2005-08-21 09:43:05Z mkaemmerer $

   OpenXP modem netcall base class
   Copyright (C) 1991-2001 Peter Mandrella
   Copyright (C) 2000-2002 OpenXP team (www.openxp.de) and M.Kiesel

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
}

{$I xpdefine.inc}

{ OpenXP modem netcall base class }
unit ncmodem;

interface

uses
  netcall,timer,objcom,progressoutput;

const
  { Log chars used in canonical log file. }
  lcCalling = '+';       { 'Calling (dest), (phonenumber)' }
  lcConnect = '=';       { '(connectstring)'/'hangup', generated by (Dis)Connect }
  lcStart = '+';         { 'starting mail transfer' }
  lcFile = '*';          { 'Send/Rcvd (file); (length)b', generated by LogRx/TxFile}
  lcStop = '+';          { 'mail transfer completed/aborted' }
  lcExit = '-';          { 'exiting', generated by Destroy}
  lcError = '%';         { 'carrier lost' }
  lcInfo = ' ';

type
  { The state we are in when sending commands to the modem }
  TModemAnswerState = (SMExpectingEcho, SMExpectingAnswer, SMDone);

  { This is the base class for all netcall types using dialup techniques.
    As ObjCOM provides the communications channel, all comm types ObjCOM
    provides are possible, even communication via IP (useful for example
    for fido over IP). }
  TModemNetcall = class(TNetcall)

  protected
    FCommObj: TCommStream;
    FTimerObj: tpTimer;
    FConnected,FActive: Boolean;
    FPhonenumbers: String;
    FGotUserBreak: Boolean;
    FLogfile: Text; FLogfileOpened: Boolean;
    ModemAnswerState: TModemAnswerState;
    FErrorMsg,FLogfileName,FCommInit,ReceivedUpToNow,ModemAnswer: String;

    FPhonenumber: String;
    FLineSpeed: Longint;
    FConnectString: String;

  protected
    procedure SLogfileName(S: String);

    { Creates FCommObj from FCommInit }
    function Activate: Boolean;

    {Process incoming bytes from modem: store in ReceivedUpToNow or move
     all bytes received yet to ModemAnswer and set ModemAnswerState
     accordingly. }
    procedure ProcessIncoming;

    {Process keypresses:
     - set timer to timeout and set FGotUserBreak to True on ESC
     - set timer to timeout on space
     - adjust timer on +/-}
    procedure ProcessKeypresses(AcceptSpace:boolean);

    {Send command to modem. Wait max TimeoutModemAnswer seconds for answer.
     Return answer if received in time; if not received store in
     ReceivedUpToNow, set ModemAnswerState<>SMDone and return empty string.}
    function SendCommand(s:string; TimeoutModemAnswer: real): String;

    {Send multiple commands separated by '\\'. Return last modem answer. See
     SendCommand for details.}
    function SendMultCommand(s:string; TimeoutModemAnswer: real): String;

    {Logs a transmitted/received file.}
    procedure LogTxRxFile(fn: string; outgoing: boolean);

  public
    {-------- Variables to initialize for modem dialing -----------------}
    {Phone numbers to dial (separated by spaces). Empty if no dialing required.}
    Phonenumbers: String;
    {Modem init string}
    CommandInit: String;
    {Modem dial prefix string}
    CommandDial: String;
    {Max dial attempts}
    MaxDialAttempts: Integer;
    {Time to wait between dial attempts}
    RedialWaitTime: Integer;
    {Connection establish timeout}
    TimeoutConnectionEstablish: Integer;
    {Modem init timeout}
    TimeoutModemInit: Integer;

    {-------- Variables available after modem dialing -----------------}
    {Phone number connected to}
    property Phonenumber: String read FPhonenumber;
    {Detected line speed (bps)}
    property LineSpeed: Longint read FLineSpeed;
    {Modem string received upon connection}
    property ConnectString: String read FConnectString;

    {------------------------- Properties ----------------------------}
    {Opens log file. Overwrites if first char is '*'.}
    property LogfileName: String read FLogfileName write SLogfileName;

    { True if comm channel initialized }
    property Active: Boolean read FActive;

    property CommObj: TCommStream read FCommObj;
    property Timer:   TPTimer     read FTimerObj;

    { True if connected to peer }
    property Connected: Boolean read FConnected;
    { Sets/reads timeout (activates on idleing of peer) }
//  property Timeout: Real read FGetTimeout write FSetTimeout;

    { True if interrupted by user }
    property GotUserBreak: Boolean read FGotUserBreak;
    property ErrorMsg: string read FErrorMsg;

    { Create with CommInit string and ProgressOutput class }
    constructor CreateWithCommInitAndProgressOutput(const aCommInit: string; aProgressOutput: TProgressOutput);
    { Create with CommObj. Intended for online calls. Active and
      Connected return true after call. }
    constructor CreateWithCommObjAndProgressOutput(p: TCommStream; aProgressOutput: TProgressOutput);
    { Disconnects if phonenumbers not empty.
      Disposes CommObj.
      Closes log file.
      Disposes ProgressOutput. }
    destructor Destroy; override;

    { Connects (= initializes comm channel and dials if necessary) }
    function Connect: boolean; virtual;

    { Logs an event in log file. See lc* log char consts.}
    procedure Log(c: Char; const s: String);
    procedure LogRxFile(fn: string);
    procedure LogTxFile(fn: string);

    { throws ENetcallHangup on no carrier, ENetcallBreak }
    { on user break (and ENetcallTimeout on timeout)     }
    procedure TestBreak;
    procedure TestTimeout;

    { Disconnects. Hangs up if phonenumbers specified. }
    procedure Disconnect; virtual;

  end;

{ Get first phone number in list and rotate list }
function GetNextPhonenumber(var Phonenumbers: string): string;
{ Count phone numbers in list }
function CountPhonenumbers(Phonenumbers: string): integer;

implementation

uses
  fileio, keys,xpglobal,sysutils,typeform,debug,xpprogressoutputwindow,osdepend;

function GetNextPhonenumber(var Phonenumbers: string): string;
var p : byte;
begin
  PhoneNumbers:=trim(Phonenumbers);
  p:=cPos(' ',Phonenumbers);
  if p=0 then result:=Phonenumbers
  else begin
    result:=LeftStr(Phonenumbers,p-1);
    Phonenumbers:=trim(mid(Phonenumbers,p))+' '+LeftStr(Phonenumbers,p-1);
  end;
end;

function CountPhonenumbers(Phonenumbers: string): integer;
var n : integer;
begin
  Phonenumbers:=trim(Phonenumbers);
  n:=1;
  while cPos(' ',Phonenumbers)>0 do begin
    Phonenumbers:=trim(mid(Phonenumbers,cpos(' ',Phonenumbers)));
    inc(n);
    end;
  result:=n;
end;

constructor TModemNetcall.CreateWithCommInitAndProgressOutput(const aCommInit: string; aProgressOutput: TProgressOutput);
begin
  inherited Create;
  FConnected:=False; FActive:=False; FErrorMsg:=''; ProgressOutput:=aProgressOutput; FCommInit:=aCommInit;
  ModemAnswerState:=SMDone; FGotUserBreak:=False; ReceivedUpToNow:=''; ModemAnswer:='';
  Phonenumbers:=''; CommandInit:='ATZ'; CommandDial:='ATD'; MaxDialAttempts:=3;
  TimeoutConnectionEstablish:=90; TimeoutModemInit:=10; RedialWaitTime:=40;
  FLogfileOpened:=False; FPhonenumber:=''; FLineSpeed:=0; FConnectString:='';
  FTimerObj:=new(TPTimer,Init);
end;

constructor TModemNetcall.CreateWithCommObjAndProgressOutput(p: TCommStream; aProgressOutput: TProgressOutput);
begin
  CreateWithCommInitAndProgressOutput('',aProgressOutput);
  FCommObj:=p; FActive:=True; FConnected:=True;
end;

destructor TModemNetcall.Destroy;
begin
  if FConnected then Disconnect;
  if FActive then begin FCommObj.Close; FCommObj.Free end;
  if Assigned(FTimerObj) then dispose(FTimerObj,Done);
  Log(lcExit,'exiting');
  if FLogfileOpened then Close(FLogfile);
  inherited destroy;
end;

procedure TModemNetcall.SLogfileName(S: String);
var Overwrite: Boolean;
begin
  Overwrite:=Copy(S,1,1)='*';
  if Overwrite then Delete(S,1,1);
  Assign(FLogfile,S);
  if Overwrite or not FileExists(s) then
    ReWrite(FLogfile)
  else
    Append(FLogfile);
  FLogfileOpened:=True;
end;

function TModemNetcall.Activate: Boolean;
begin
  if not FActive then begin 
    FCommObj:=CommInit(FCommInit);
    FActive:=Assigned(FCommObj);
  end;
  if not FActive then begin
    FErrorMsg:=ObjCOM.ErrorStr;
    Output(mcError,'%s',[FErrorMsg]);
    Log(lcError,FErrorMsg);
    end;
  result:=FActive;
end;

procedure TModemNetcall.ProcessIncoming;
var c : char;
begin
  if FCommObj.CharAvail then begin
    c:=FCommObj.GetChar;
    if (c=#13) or (c=#10) then begin
      case ModemAnswerState of
        SMExpectingEcho: begin
                           ModemAnswerState:=SMExpectingAnswer;
                           ReceivedUpToNow:='';
                         end;
        SMExpectingAnswer: begin
                             if ReceivedUpToNow<>'' then begin
                               ModemAnswerState:=SMDone;
                               ModemAnswer:=ReceivedUpToNow; ReceivedUpToNow:='';
                               DebugLog('ncmodem','Modem answer: "'+ModemAnswer+'"',DLDebug);
                             end;
                           end;
      end;
    end else if c<>#0 then ReceivedUpToNow:=ReceivedUpToNow+c;
  end else SysDelay(2);
end;

procedure TModemNetcall.ProcessKeypresses(AcceptSpace:boolean);
var c : char;
begin
  if keys.keypressed then begin
    c:=keys.readkey;

    if c=#0 then case keys.readkey of
      #243 {mausunright}: c:=#27;
      #241 {mausunleft}:  c:=' ';
      #248 {mauswheelup}: c:='+';
      #249 {mauswheeldn}: c:='-';
    end;

    case c of
      #27 : begin
              FTimerObj.SetTimeout(0); ModemAnswerState:=SMDone; ReceivedUpToNow:='';
              DebugLog('ncmodem','User break',DLWarning); FGotUserBreak:=true;
            end;
      '+' : FTimerObj.SetTimeout(FTimerObj.SecsToTimeout+1);
      '-' : if FTimerObj.SecsToTimeout>1 then FTimerObj.SetTimeout(FTimerObj.SecsToTimeout-1);
      ' ' : if AcceptSpace then FTimerObj.SetTimeout(0);
    end;
  end;
end;

function TModemNetcall.SendCommand(s: string; TimeoutModemAnswer: real): String;
var p : byte; EchoTimer: tTimer;
begin
  DebugLog('ncmodem','SendCommand: "'+s+'"',DLDebug);
  FCommObj.PurgeInBuffer; s:=trim(s);
  if s<>'' then begin {Nicht-leerer Modembefehl; Tilde im Befehl bedeutet ca. 1 Sec Pause}
    repeat
      p:=cpos('~',s);
      if p>0 then begin
        FCommObj.SendString(LeftStr(s,p-1),false);
        delete(s,1,p); SysDelay(1000);
      end;
    until p=0;
    FCommObj.SendString(s+#13,false);
    EchoTimer.Init; EchoTimer.SetTimeout(TimeoutModemAnswer); ReceivedUpToNow:=''; ModemAnswerState:=SMExpectingEcho;
    repeat
      ProcessIncoming; ProcessKeypresses(false);
    until EchoTimer.Timeout or (ModemAnswerState=SMDone); {Warte auf Antwort}
    if EchoTimer.Timeout then ModemAnswer:='';
    SysDelay(200); EchoTimer.Done;
    SendCommand:=ModemAnswer; DebugLog('ncmodem','SendCommand: Got modem answer "'+ModemAnswer+'"',DLDebug);
  end;
end;

function TModemNetcall.SendMultCommand(s: string; TimeoutModemAnswer: real): String;
var p : byte; cmd: String;
begin
  DebugLog('ncmodem','SendMultCommand: "'+s+'"',DLDebug);
  while (length(trim(s))>1) do begin
    p:=pos('\\',s);
    if p=0 then p:=length(s)+1;
    cmd:=trim(LeftStr(s,p-1));
    SendMultCommand:=SendCommand(cmd,TimeoutModemAnswer);
    s:=trim(mid(s,p+2));
    ProcessKeypresses(false);
  end;
end;

function TModemNetcall.Connect: boolean;

  function Bauddetect(ConnectString: String): Longint;
  var p: byte; b: longint;
  begin
    p:=1;
    while(p<=length(ConnectString))and((ConnectString[p]<'0')or(ConnectString[p]>'9'))do inc(p);
    delete(ConnectString,1,p-1);
    p:=1;
    while(p<=length(ConnectString))and(ConnectString[p]>='0')and(ConnectString[p]<='9')do inc(p);
    b:=ival(LeftStr(ConnectString,p-1));
    if(b<300)or(115200 mod b<>0)then Bauddetect:=0 else Bauddetect:=b;
  end;

type tStateDialup= (SDInitialize,SDSendDial,SDWaitForConnect,SDWaitForNextCall,SDModemAnswer,SDConnect,SDNoConnect,SDUserBreak);

var
  StateDialup: tStateDialup;
  iDial: Integer;
  CurrentPhonenumber: String;

begin
  if not FActive then begin
    Output(mcVerbose,'Opening comm channel',[0]);
    result:=Activate;
    if not result then exit;
    end;
  if Phonenumbers='' then begin
    Log(lcConnect,'CONNECT');
    FConnected:=true; result:=true; exit
    end;
  FGotUserBreak := false;
  DebugLog('ncmodem','Dialup: Numbers "'+Phonenumbers+'", Init "'+CommandInit+'", Dial "'+CommandDial+'", MaxDialAttempts '+
                   Strs(MaxDialAttempts)+', ConnectionTimeout '+Strs(TimeoutConnectionEstablish)+', RedialWait '+Strs(RedialWaitTime),DLInform);
  StateDialup:=SDInitialize; iDial:=0; result:=False;

  while StateDialup<=SDWaitForNextCall do begin
    case StateDialup of
      SDInitialize: begin
                      Output(mcInfo,'Init modem',[0]);
                      FTimerObj.SetTimeout(TimeoutModemInit);
                      if CommandInit='' then begin
                        FCommObj.SendString(#13,False); SysDelay(150);
                        FCommObj.SendString(#13,False); SysDelay(300);
                        SendCommand('AT',1);
                      end else
                        SendMultCommand(CommandInit,TimeoutModemInit);
                      if FTimerObj.Timeout then
                        StateDialup:=SDNoConnect
                      else
                        StateDialup:=SDSendDial;
                    end;
      SDSendDial: begin
                    inc(iDial); FPhonenumber:=GetNextPhonenumber(Phonenumbers);
                    Output(mcInfo,'Dial %s try %d',[FPhonenumber,iDial]);
                    CurrentPhonenumber:=FPhonenumber;
                    while cpos('-',CurrentPhonenumber)>0 do delete(CurrentPhonenumber,cpos('-',CurrentPhonenumber),1);
                    SendMultCommand(CommandDial+CurrentPhonenumber,1); {Gegenstelle anwaehlen}
                    StateDialup:=SDWaitForConnect;
                  end;
      SDWaitForConnect: begin
                          FTimerObj.SetTimeout(TimeoutConnectionEstablish);
                          TProgressOutputWindow(ProgressOutput).TimerDisplay:=mwTimeout;
                          TProgressOutputWindow(ProgressOutput).TimerToUse:=FTimerObj;
                          repeat
                            ProcessIncoming; ProcessKeypresses(false);
                            Output(mcVerbose,'',[0]);
                          until FTimerObj.Timeout or(ModemAnswerState=SMDone);
                          TProgressOutputWindow(ProgressOutput).TimerDisplay:=mwElapsedTime;
                          TProgressOutputWindow(ProgressOutput).TimerToUse:=@TProgressOutputWindow(ProgressOutput).Timer;
                          result:=False;
                          if not FTimerObj.Timeout then begin
                            {Kein Timeout, kein Userbreak: Vermutlich Connect oder Busy.}
                            Output(mcInfo,'%s',[ModemAnswer]);
                            SysDelay(200);
                            if LeftStr(ModemAnswer,7)='CARRIER' then ModemAnswer:='CONNECT'+mid(ModemAnswer,8);
                            if LeftStr(ModemAnswer,7)='CONNECT' then begin
                              {Connect!}
                              TProgressOutputWindow(ProgressOutput).Timer.Start;
                              StateDialup:=SDConnect; result:=True;
                              FConnectString:=ModemAnswer; FConnected:=True;
                              FLineSpeed:=Bauddetect(FConnectString);
                              Log(lcConnect,FConnectString);
                              if not FCommObj.Carrier then SysDelay(500);  { falls Carrier nach CONNECT kommt }
                              if not FCommObj.Carrier then SysDelay(1000);
                            end
                          end;
                          if not result then begin {Timeout, Userbreak, Busy oder aehnliches}
                            Output(mcInfo,'No connect',[0]);
                            FPhonenumber:='';
                            FCommObj.SendString(#13,False); SysDelay(1000); {ggf. noch auflegen}
                            StateDialup:=SDWaitForNextCall;
                          end;
              end;
      SDWaitForNextCall: begin
                           FTimerObj.SetTimeout(RedialWaitTime);
                           Output(mcInfo,'Wait for next dial attempt',[0]);
                           if iDial<MaxDialAttempts then begin
                             TProgressOutputWindow(ProgressOutput).TimerDisplay:=mwTimeout;
                             TProgressOutputWindow(ProgressOutput).TimerToUse:=FTimerObj;
                             repeat
                               Output(mcVerbose,'',[0]);
                               ProcessIncoming; ProcessKeypresses(true);
                               if Pos('RING',ModemAnswer)<>0 then begin
                                 Output(mcInfo,'Ring detected',[0]);
                                 ModemAnswerState:=SMExpectingAnswer; FTimerObj.SetTimeout(RedialWaitTime);
                               end;
                             until FTimerObj.Timeout;
                             TProgressOutputWindow(ProgressOutput).TimerDisplay:=mwElapsedTime;
                             TProgressOutputWindow(ProgressOutput).TimerToUse:=@TProgressOutputWindow(ProgressOutput).Timer;
                             StateDialup:=SDInitialize;
                           end else StateDialup:=SDNoConnect;
                         end;
    end;
    ProcessKeypresses(true);
    if FGotUserBreak then begin Output(mcInfo,'Got user break',[0]); exit end;
  end;
end;

procedure TModemNetcall.Log(c: Char; const s: String);
begin
  if FLogfileOpened then
    writeln(FLogfile,c,' ',FormatDateTime('hh":"nn":"ss',Now),'  ',s);
  DebugLog('ncmodem','Log: '+c+' '+s,dlInform);
end;

procedure TModemNetcall.LogTxRxFile(fn: string; outgoing: boolean);
var FileSize,FileHandle: LongInt;
begin
  FileHandle:=FileOpen(fn,fmOpenRead);
  if (FileHandle>0)then begin
    FileSize:=FileSeek(FileHandle,0,fsFromEnd);
    FileClose(FileHandle);
    if outgoing then
      Log(lcFile,'Sent '+ExtractFileName(fn)+', '+IntToStr(FileSize)+'b')
    else
      Log(lcFile,'Rcvd '+ExtractFileName(fn)+', '+IntToStr(FileSize)+'b');
    end;
end;

procedure TModemNetcall.LogRxFile(fn: string);
begin LogTxRxFile(fn,false) end;

procedure TModemNetcall.LogTxFile(fn: string);
begin LogTxRxFile(fn,true) end;

procedure TModemNetcall.Disconnect;
var i : integer;
begin
  if FConnected then Log(lcConnect,'hangup');
  if FPhonenumber<>'' then begin
    Output(mcInfo,'Hanging up',[0]);
    DebugLog('ncmodem','Hangup',DLInform);
    FCommObj.PurgeInBuffer; FCommObj.SetDTR(False);
    SysDelay(500); for i:=1 to 6 do if(not FCommObj.IgnoreCD)and FCommObj.Carrier then SysDelay(500);
    FCommObj.SetDTR(True); SysDelay(500);
    if FCommObj.ReadyToSend(3)then begin
      FCommObj.SendString('+++',False);
      for i:=1 to 4 do if((not FCommObj.IgnoreCD)and FCommObj.Carrier)then SysDelay(500);
      SysDelay(100);
    end;
    if FCommObj.ReadyToSend(6)then
      FCommObj.SendString('AT H0'#13,false);
    end;
  FConnected:=False;
end;

{ throws ENetcallHangup on no carrier, ENetcallBreak on user break }
procedure TModemNetcall.TestBreak;
begin
  if not FCommObj.Carrier then
    raise ENetcallHangup.Create('carrier lost');

  if not FGotUserBreak then
    if keypressed and (readkey=#27) then
    begin
      FGotUserBreak:=true;
      Log(lcExit,'User break.');
      Output(mcInfo,'User break - aborting...',[0]);
    end;

  if FGotUserBreak then
    raise ENetcallBreak.Create('user break');
end;

procedure TModemNetcall.TestTimeout;
begin
  if FTimerObj.Timeout then
    raise ENetcallTimeout.Create('timeout');
  TestBreak;
end;

end.
